#include <gst/gst.h>
#include <gst/video/videooverlay.h>
#import "GStreamerBackend.h"
#import "VideoStreamBackendDelegate.h"
#import "VSLogging.h"

@implementation GStreamerBackend
{
    NSUInteger _windowHandleSetInstances;
}

#pragma mark public interface methods

-(id) initWithDelegate:(id)delegate streamId:(NSString *)streamId displayView:(EaglUIView *)displayView
{
    TRC_ENTRY;
    if (self = [super init])
    {
        gst_debug_set_default_threshold(GST_LEVEL_INFO);
        _delegate = delegate;
        _streamId = streamId;
        _displayView = displayView;
        _isActive = NO;
        _windowHandleSetInstances = 0;
    }
    TRC_EXIT;
    return self;
}

- (void) start
{
    TRC_ENTRY;
    if (!self.isActive)
    {
        __weak GStreamerBackend *weakSelf = self;
        dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0), ^
        {
            [weakSelf buildAndRunPipeline];
        });
    }
    TRC_EXIT;
}

-(void) stop
{
    TRC_ENTRY;
    if (self.isActive)
    {
        gst_element_send_event(_pipeline, gst_event_new_eos());
    }
    TRC_EXIT;
}

/**
 * Utility method to get the current gstreamer version, so we can always be sure
 * what we're working with.
 */
-(NSString*) getGStreamerVersion
{
    char *version_utf8 = gst_version_string();
    NSString *version_string = [NSString stringWithUTF8String:version_utf8];
    g_free(version_utf8);
    return version_string;
}

#pragma mark utility methods

/**
 * Build a pipeline using the parse launch method
 */
-(NSString *) buildPipelineUsingParse
{
    TRC_ENTRY;
    NSString *initialError = nil;
    
    // pipeline def
    NSString *source = @"videotestsrc is-live=1 horizontal-speed=1 name=zeVideoSource";
    source = [source stringByAppendingString:@" ! video/x-raw, width=640, height=480"];
    source = [source stringByAppendingString:@" ! videoconvert"];
    source = [source stringByAppendingString:@" ! autovideosink sync=false"];
    
    // build pipeline
    GError *pipelineErr = NULL;
    _pipeline = gst_parse_launch([source cStringUsingEncoding:NSUTF8StringEncoding], &pipelineErr);
    
    // check for error
    if (pipelineErr != NULL)
    {
        initialError = @"Unable to build pipeline using gst_parse_launch";
        LOG_ERR(initialError, nil);
        
        gchar *errMsg = g_strdup_printf ("Pipeline build error: %s", pipelineErr->message);
        LOG_ERR([NSString stringWithUTF8String:errMsg], nil);
        g_free(errMsg);
    }
    
    TRC_EXIT;
    return initialError;
}

/**
 * Master method for building and running the pipeline.
 */
-(void) buildAndRunPipeline
{
    TRC_ENTRY;
    
    // build pipeline
    NSString *initialError = [self buildPipelineUsingParse];
    
    // set pipeline state to ready to accept a window handle
    if (initialError == nil)
    {
        gst_element_set_state(_pipeline, GST_STATE_READY);
        GstElement *videoSink = gst_bin_get_by_interface(GST_BIN(_pipeline), GST_TYPE_VIDEO_OVERLAY);
        if (!videoSink)
        {
            initialError = @"Could not retrieve overlay video sink";
            LOG_ERR(initialError, nil);
        }
        else
        {
            // creates a new UIView for rendering video display
            // if this pipeline is started multiple times
            if (_windowHandleSetInstances > 0)
                [self rebuildUIView];
            
            guintptr videoHandle = (guintptr)(id)_displayView;
            gst_video_overlay_set_window_handle(GST_VIDEO_OVERLAY(videoSink), videoHandle);
            g_object_unref(videoSink);
            
            _windowHandleSetInstances++;
        }
    }
    
    // set pipeline into playing state
    if (initialError == nil)
    {
        if (gst_element_set_state(_pipeline, GST_STATE_PLAYING) == GST_STATE_CHANGE_FAILURE)
        {
            initialError = @"Unable to set pipeline in playing state";
            LOG_ERR(initialError, nil);
            gst_element_set_state(_pipeline, GST_STATE_NULL);
        }
    }
    
    // update caller we started and run main loop
    if (initialError == nil)
    {
        [self runPipeline];
    }
    else
    {
        // alert the caller if we did not start successfully
        if (_delegate && [_delegate respondsToSelector:@selector(streamRequestFailed:reason:)])
        {
            [_delegate streamRequestFailed:self reason:initialError];
        }
        
        [self pipelineCleanup];
    }
    
    TRC_EXIT;
}

/**
 * The running of the pipeline is broken out into a separate method so it can be executed
 * on any given queue.
 */
-(void) runPipeline
{
    //
    // start a main loop to keep the pipeline running
    
    [self setActive:YES];
    
    // create our own GLib main context and make it the default one
    _context = g_main_context_new ();
    g_main_context_push_thread_default(_context);
    
    // create and run a new loop
    _mainLoop = g_main_loop_new (_context, FALSE);
    
    // add a watch on the bus
    // this must happen after the context and main loop are created
    GstBus *bus = gst_pipeline_get_bus(GST_PIPELINE(_pipeline));
    gst_bus_add_watch(bus, gstcBusMessage, (__bridge gpointer)(self));
    g_object_unref(bus);
    
    LOG_DEBUG(@"Entering main loop");
    g_main_loop_run (_mainLoop);
    LOG_DEBUG(@"Exited main loop");
    
    // loop and context cleanup
    g_main_loop_unref (_mainLoop);
    g_main_context_pop_thread_default(_context);
    g_main_context_unref (_context);
    
    //
    // will fall-through when the loop is exited

    [self setActive:NO];
    
    [self pipelineCleanup];
}

-(void) pipelineCleanup
{
    // cleanup our failed pipeline
    gst_element_set_state(_pipeline, GST_STATE_NULL);
    gst_object_unref (_pipeline);
    _pipeline = NULL;
}

/**
 * When the pipeline is rebuilt, we need a new UIView for the open gl rendering.
 * Handle this internally to rebuild a new view to match the existing one.
 */
-(void) rebuildUIView
{
    LOG_DEBUG(@"*** rebuilding display backend UIView");
    
    dispatch_sync(dispatch_get_main_queue(), ^{
        
        EaglUIView *currentView = _displayView;
        _displayView = [[EaglUIView alloc] initWithFrame:currentView.frame];
        
        // if our view has gesture recognizers, add them to the new view
        if (currentView.gestureRecognizers.count > 0)
        {
            for (UIGestureRecognizer *recognizer in currentView.gestureRecognizers)
            {
                [currentView removeGestureRecognizer:recognizer];
                [_displayView addGestureRecognizer:recognizer];
            }
        }
        
        // if view is showing, show our new view as well
        if (currentView.superview != nil)
        {
            UIView *superView = currentView.superview;
            [currentView removeFromSuperview];
            [superView addSubview:_displayView];
        }
    });
}

/**
 * Utility method to check if a bus message source object is our pipeline.
 */
-(BOOL) objectIsPipeline:(GstObject *)object
{
    if (object == GST_OBJECT(_pipeline))
        return YES;
    
    return NO;
}

#pragma mark delegate utilities

/**
 * Post a message to the delegate
 */
-(void)setMessage:(NSString *) message
{
    if(_delegate && [_delegate respondsToSelector:@selector(streamMessage:message:)])
    {
        [_delegate streamMessage:self message:message];
    }
}

/**
 * update the active state, and trigger the active delegate if one has been defined
 */
-(void)setActive:(BOOL)isActive
{
    BOOL activeChanged = _isActive != isActive;
    _isActive = isActive;
    
    if (activeChanged)
    {
        NSString *activeDescription = isActive ? @"ACTIVE" : @"INACTIVE";
        LOG_DEBUG(@"(%@) changed to %@", _streamId, activeDescription);
        if (_delegate && [_delegate respondsToSelector:@selector(streamActiveChanged:isActive:)])
        {
            [_delegate streamActiveChanged:self isActive:_isActive];
        }
    }
}

#pragma mark gstreamer bus callbacks

-(void) gstLeaveMainLoop
{
    LOG_DEBUG(@"Leaving gst main loop");
    g_main_loop_quit(_mainLoop);
}

-(void) gstElementMessage:(GstBus *)bus message:(GstMessage *)msg
{
    TRC_ENTRY;
    TRC_EXIT;
}

-(void) gstPipelineStateChanged:(GstState)oldState newState:(GstState)newState pendingState:(GstState)pendingState
{
    TRC_ENTRY;
    TRC_EXIT;
}

/**
 * Pipeline element error callback
 */
static void gstcEos (GstBus *bus, GstMessage *msg, gpointer userData)
{
    GStreamerBackend *self = (__bridge GStreamerBackend *)userData;
    LOG_DEBUG(@"EOS received on bus");
    [self gstLeaveMainLoop];
}

/**
 * Pipeline element error callback
 */
static void gstcError (GstBus *bus, GstMessage *msg, gpointer userData)
{
    GStreamerBackend *self = (__bridge GStreamerBackend *)userData;
    
    GError *err = NULL;
    gchar *dbg;
    
    gst_message_parse_error(msg, &err, &dbg);
    gst_object_default_error(msg->src, err, dbg);
    
    gchar *errMsg = g_strdup_printf ("Error received from element %s: %s", GST_OBJECT_NAME (msg->src), err->message);
    [self setMessage:[NSString stringWithUTF8String:errMsg]];
    g_free(errMsg);
    
    g_error_free(err);
    g_free(dbg);
    
    [self gstLeaveMainLoop];
}

/**
 * Pipeline element state changed callback
 */
static void gstcElementStateChanged (GstBus *bus, GstMessage *msg, gpointer userData)
{
    GStreamerBackend *self = (__bridge GStreamerBackend *)userData;
    
    // only pay attention to messages coming from the pipeline, not its children
    if ([self objectIsPipeline:GST_MESSAGE_SRC (msg)])
    {
        GstState oldState, newState, pendingState;
        gst_message_parse_state_changed (msg, &oldState, &newState, &pendingState);
        
        gchar *message = g_strdup_printf("pipeline (%s) state changed to %s", [self.streamId cStringUsingEncoding:NSUTF8StringEncoding], gst_element_state_get_name(newState));
        LOG_DEBUG([NSString stringWithUTF8String:message], nil);
        g_free (message);
        
        [self gstPipelineStateChanged:oldState newState:newState pendingState:pendingState];
    }
}

/**
 * Pipeline element message callback
 */
static void gstcElementMesage(GstBus *bus, GstMessage *msg, gpointer userData)
{
    GStreamerBackend *self = (__bridge GStreamerBackend *)userData;
    [self gstElementMessage:bus message:msg];
}

/**
 * Primary bus message callback
 */
static gboolean gstcBusMessage (GstBus *bus, GstMessage *msg, gpointer userData)
{
    switch (GST_MESSAGE_TYPE(msg))
    {
        case GST_MESSAGE_EOS:
        {
            gstcEos(bus, msg, userData);
            break;
        }
            
        case GST_MESSAGE_ERROR:
        {
            gstcError(bus, msg, userData);
            break;
        }
            
        case GST_MESSAGE_STATE_CHANGED:
        {
            gstcElementStateChanged(bus, msg, userData);
            break;
        }
            
        case GST_MESSAGE_ELEMENT:
        {
            gstcElementMesage(bus, msg, userData);
            break;
        }
            
        default:
        {
            //LOG_TRACE(@"unhandled gst message type: %s", GST_MESSAGE_TYPE_NAME(msg));
            break;
        }
    }
    return TRUE;
}

@end

